static char rcsid[] = "@(#)$Id: md5_wrapper.c,v 1.3 2020/05/31 11:57:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.3 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"digest");


#include "md5_global.h"
#include "md5.h"
#include "digest_imp.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}



S_(digest_proc_alloc_ctx md5_alloc_ctx)
static void md5_alloc_ctx P_((struct digest_proc *dp));

S_(digest_proc_free_ctx md5_free_ctx)
static void md5_free_ctx P_((struct digest_proc *dp));

S_(digest_proc_init_ctx md5_init_ctx)
static void md5_init_ctx  P_((struct digest_proc *dp));

S_(digest_proc_update_ctx md5_update_ctx)
static void md5_update_ctx  P_((struct digest_proc *dp,
				const unsigned char data[],
				const size_t        data_len));

S_(digest_proc_final_ctx md5_final_ctx)
static void md5_final_ctx  P_((struct digest_proc *dp,
			       unsigned char digest_buffer[],
			       size_t        digest_buffer_len));

#define MD5_DIGEST_LEN 16

static struct tests {
    const char *test_case;
    const unsigned char correct[MD5_DIGEST_LEN];
} md5_tests[]  UNUSED_VAROK = {
    /* Tests from RFC 1321 */
    
    { "",  { 0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04, 0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e } },
    { "a", { 0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8, 0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61 } },
    { "abc", { 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0, 0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72 } },
    { "message digest",
      { 0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d, 0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0 } },
    { "abcdefghijklmnopqrstuvwxyz",
      { 0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00, 0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b } },
    { "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
      { 0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5, 0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f } },
    { "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
      { 0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55, 0xac, 0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a } }
};


static void debug_result P_((const unsigned char buffer[MD5_DIGEST_LEN]));
static void debug_result(buffer)
     const unsigned char buffer[MD5_DIGEST_LEN];
{
    DEBUG_CHECK_INIT(Debug);
    
    if (DEBUG_CHECK(Debug,10)) {
	size_t x;
	
	for (x = 0; x < MD5_DIGEST_LEN; x++) {
#if DEBUG
	    debug_action_call(&Debug,"%02X",buffer[x]);
#endif
	}
    }   
}

static enum md5_test {
    md5_test_failed = -1,
    md5_test_none   = 0,
    md5_test_succeed
} md5_test_suite P_((struct digest_proc_type *impl));
static enum md5_test md5_test_suite(impl)
     struct digest_proc_type *impl;
{
    struct digest_proc X;

    enum md5_test ret = md5_test_none;

    int i;
    
    if (DIGEST_PROC_TYPE_magic != impl->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_test_suite",
	      "Bad magic number (digest_proc_type)",0);

    if (MD5_DIGEST_LEN != impl->digest_len) {
	DPRINT(Debug,1,(&Debug,
			"md5_test_suite=md5_test_failed: impl=%p MD5_DIGEST_LEN (%d) != impl->digest_len (%zu)\n",
			impl,MD5_DIGEST_LEN,impl->digest_len));

        ret= md5_test_failed;
	goto fail;
    }
	
    bzero(&X,sizeof X);

    X.magic =  DIGEST_PROC_magic;
    X.digest_type = impl;
    X.mode        = digest_proc_initial;
    X.context.dummy = NULL;
    X.digest_result = NULL;
    
    impl->alloc_ctx(&X);

    for (i = 0; i < (sizeof md5_tests) / sizeof (md5_tests[0]); i ++) {
	size_t len = strlen(md5_tests[i].test_case);
	unsigned char result[MD5_DIGEST_LEN];
	
	impl->init_ctx(&X);
	impl->update_ctx(&X,cs2us(md5_tests[i].test_case),len);

	impl->final_ctx(&X,result,sizeof result);

	DPRINT(Debug,10,(&Debug,
			 "md5_test_suite: impl=%p test %d MD5 \"%s\" (len %zu) = \thex ",
			 impl,i,md5_tests[i].test_case,len));

	debug_result(result);

	DPRINT(Debug,10,(&Debug," "));

	if (0 == memcmp(result,md5_tests[i].correct,MD5_DIGEST_LEN)) {
	    DPRINT(Debug,10,(&Debug,"OK"));
	} else {
	    DPRINT(Debug,10,(&Debug,"FAIL != hex "));
	    debug_result(md5_tests[i].correct);
	    ret = md5_test_failed;
	}
	DPRINT(Debug,10,(&Debug,"\n"));

    }

    if (md5_test_none == ret) {
	ret = md5_test_succeed;

	DPRINT(Debug,10,(&Debug,
			 "md5_test_suite=md5_test_succeed: All tests succeed\n"));
    }

    impl->free_ctx(&X);

 fail:

    DPRINT(Debug,10,(&Debug,
		     "md5_test_suite=%d",ret));
    switch (ret) {
    case md5_test_failed:   DPRINT(Debug,10,(&Debug," md5_test_failed"));  break;
    case md5_test_none:     DPRINT(Debug,10,(&Debug," md5_test_none"));    break;
    case md5_test_succeed:  DPRINT(Debug,10,(&Debug," md5_test_succeed")); break;
    }    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

static struct digest_proc_type  md5_digest = {
    DIGEST_PROC_TYPE_magic,
    MD5_DIGEST_LEN,
    md5_alloc_ctx,
    md5_free_ctx,
    md5_init_ctx,
    md5_update_ctx,
    md5_final_ctx
};

static enum md5_test md5_test_result = md5_test_none;

const struct digest_proc_type * md5_available()
{
    const struct digest_proc_type *ret = NULL;
    
    if (md5_test_none == md5_test_result) {
	DPRINT(Debug,10,(&Debug,"md5_available: Need run md5_test_suite\n"));
	       
	md5_test_result = md5_test_suite(&md5_digest);
    }

    switch (md5_test_result) {
    case md5_test_failed:
	DPRINT(Debug,10,(&Debug,"md5_available: md5_test_suite returns md5_test_failed\n"));
	break;
    case md5_test_none:
	DPRINT(Debug,10,(&Debug,"md5_available: md5_test_suite returns md5_test_none\n"));
	break;
    case md5_test_succeed:
	ret = &md5_digest;
	break;
    }

    return ret;
}

struct digest_proc * digest_proc_md5()
{
    struct digest_proc *ret = NULL;
    
    if (md5_test_none == md5_test_result) {
	DPRINT(Debug,10,(&Debug,"digest_proc_md5: Need run md5_test_suite\n"));
	       
	md5_test_result = md5_test_suite(&md5_digest);
    }
    
    switch (md5_test_result) {
    case md5_test_failed:
	DPRINT(Debug,10,(&Debug,"digest_proc_md5: md5_test_suite returns md5_test_failed\n"));
	break;
    case md5_test_none:
	DPRINT(Debug,10,(&Debug,"digest_proc_md5: md5_test_suite returns md5_test_none\n"));
	break;
    case md5_test_succeed:
	ret = malloc_digest_proc(&md5_digest);
	break;
    }
    
    return ret;
}

/* Implementation of wrapper */

static void md5_alloc_ctx(dp)
     struct digest_proc *dp;
{
    if (DIGEST_PROC_magic != dp->magic ||
	&md5_digest       != dp->digest_type)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_alloc_ctx",
	      "Bad magic number (md5_alloc_ctx) or digest_type",0);

    if (dp->context.md5)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_alloc_ctx",
	      "context already alloced",0);
	
    dp->context.md5 =  safe_zero_alloc(sizeof (* (dp->context.md5)));
    /* No magic numbers ! */
        
}

static void md5_free_ctx(dp)
     struct digest_proc *dp;
{
    if (DIGEST_PROC_magic != dp->magic ||
	&md5_digest       != dp->digest_type)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_free_ctx",
	      "Bad magic number (md5_alloc_ctx) or digest_type",0);

    if (! dp->context.md5)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_free_ctx",
	      "context not alloced",0);

    free(dp->context.md5);
    dp->context.md5 = NULL;
}

static void md5_init_ctx(dp)
     struct digest_proc *dp;
{
    if (DIGEST_PROC_magic != dp->magic ||
	&md5_digest       != dp->digest_type)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_init_ctx",
	      "Bad magic number (md5_alloc_ctx) or digest_type",0);

    if (! dp->context.md5)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_init_ctx",
	      "context not alloced",0);

    MD5Init(dp->context.md5);
}

static void md5_update_ctx(dp,data,data_len)
     struct digest_proc *dp;
     const unsigned char data[];
     const size_t        data_len;
{
    if (DIGEST_PROC_magic != dp->magic ||
	&md5_digest       != dp->digest_type)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_update_ctx",
	      "Bad magic number (md5_alloc_ctx) or digest_type",0);

    if (! dp->context.md5)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_update_ctx",
	      "context not alloced",0);

    MD5Update(dp->context.md5,data,data_len);
}

static void md5_final_ctx(dp,digest_buffer,digest_buffer_len)
     struct digest_proc *dp;
     unsigned char digest_buffer[];
     size_t        digest_buffer_len;
{
    if (DIGEST_PROC_magic != dp->magic ||
	&md5_digest       != dp->digest_type)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_final_ctx",
	      "Bad magic number (md5_alloc_ctx) or digest_type",0);

    if (! dp->context.md5)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_final_ctx",
	      "context not alloced",0);

    if (digest_buffer_len != MD5_DIGEST_LEN)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "md5_final_ctx",
	      "Bad buffer size",0);

    MD5Final(digest_buffer,dp->context.md5);
}




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
