static char rcsid[] = "@(#)$Id: file_changes.c,v 2.1 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.1 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"addr");

const struct file_changes NULL_file_changes = FILE_CHANGES_INIT;

/* st == NULL resets file_information */

void stat_to_file_changes(st,file_information)
     const struct stat *st;
     struct file_changes *file_information;
{
    if (FILE_CHANGES_magic != file_information->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"stat_to_file_changes",
	      "Bad magic number (file_changes)",0);

    if (sizeof (*file_information) != file_information->self_size) {
	    
	DPRINT(Debug,1,(&Debug,
			"stat_to_file_changes: size mismatch file_changes size %zu != self size %zu\n",
			sizeof (*file_information),file_information->self_size));

	return;
    }
    
    *file_information = NULL_file_changes;
    
    if (st) {
	file_information->dev   = st->st_dev;
	file_information->ino   = st->st_ino;
	file_information->size  = st->st_size;
	file_information->mtime = st->st_mtime;
	file_information->valid = 1;
    }    
}

enum file_changes_result test_file_changes(filename,old_file,errno_res)
     const char                * filename;
     const struct file_changes * old_file;
     int                       * errno_res;
{
    enum file_changes_result ret = file_changes_error;
    
    struct stat filename_stat;
    int have_filename_stat = 0;
    enum syscall_status  r;

    
    if (FILE_CHANGES_magic != old_file->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"test_file_changes",
	      "Bad magic number (file_changes)",0);
    
    if (errno_res)
	* errno_res = 0;

    if (sizeof (*old_file) != old_file->self_size) {
	
	DPRINT(Debug,1,(&Debug,
			"test_file_changes: size mismatch file_changes size %zu != self size %zu\n",
			sizeof (*old_file),old_file->self_size));
	ret = file_changes_error;
	goto fail;
    }
    
    r = stat(filename,&filename_stat);
    switch (r) {
	char *X;
	int err;
    case syscall_success:
	DPRINT(Debug,14,(&Debug,
			 "test_file_changes: stat %s succeed: dev %ld ino %ld size %ld modified %ld",
			 filename,
			 (long)filename_stat.st_dev,
			 (long)filename_stat.st_ino,
			 (long)filename_stat.st_size,
			 (long)filename_stat.st_mtime));
	X = ctime(& (filename_stat.st_mtime));
	if (X) { /* ctime() includes newline */
	    DPRINT(Debug,14,(&Debug," -- %s",X));
	} else {
	    DPRINT(Debug,14,(&Debug,"\n"));
	}
	
	have_filename_stat = 1;
	
	break;
    case syscall_error:
	err = errno;
	
	DPRINT(Debug,14,(&Debug,
			 "test_file_changes: stat %s: %s\n",
			 filename,strerror(err)));

	ret = file_changes_error;
	if (errno_res)
	    * errno_res = err;
	    
	goto fail;
    }

    if (! old_file->valid || !	have_filename_stat) {
	DPRINT(Debug,14,(&Debug,
			 "test_file_changes: %s: no information -- assumed changed\n", filename));
	ret = have_file_changes;
    } else if (filename_stat.st_dev   == old_file->dev  &&
	       filename_stat.st_ino   == old_file->ino  &&
	       filename_stat.st_size  == old_file->size &&
	       filename_stat.st_mtime == old_file->mtime) {

	DPRINT(Debug,14,(&Debug,
			 "test_file_changes: %s: No changes\n",
			 filename));
	ret = 	no_file_changes;
    } else {
	DPRINT(Debug,14,(&Debug,
			 "test_file_changes: %s changed.\n",
			 filename));
	ret = have_file_changes;
    }

 fail:
    DPRINT(Debug,14,(&Debug,
		     "test_file_changes=%d",ret));
    switch (ret) {
    case file_changes_error: DPRINT(Debug,14,(&Debug," file_changes_error")); break;
    case no_file_changes:    DPRINT(Debug,14,(&Debug," no_file_changes"));    break;
    case have_file_changes:  DPRINT(Debug,14,(&Debug," have_file_changes"));  break;
    }
    if (errno_res && * errno_res) {
	DPRINT(Debug,14,(&Debug,"; error %s",
			 strerror(* errno_res)));
    }
    
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
