static char rcsid[] = "@(#)$Id: addr_list.c,v 2.9 2016/03/21 20:26:13 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.9 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"addr");

#define ADDR_list_magic      0xE800

struct addr_list {
    unsigned short magic;        /*  ADDR_list_magic */

    int              group_count;
    struct string ** groups;
    
    struct addr_list_item {
	int               groupidx;   /* -1 == not in group */
	struct address  * addr;
    }               * addrs;
    int               addr_count;
    int               alloced;       

};

/* ------------------------------------------------------------------------------ */

void free_addr_list(list)
     struct addr_list **list;
{
    if (ADDR_list_magic != (*list)->magic)
	panic("URL PANIC",__FILE__,__LINE__,"free_addr_list",
	      "Bad magic number",0);

    if ((*list)->groups) {
	int i;

	for (i = 0; i < (*list)->group_count; i++)
	    if ((*list)->groups[i])
		free_string(& ((*list)->groups[i]));

	free((*list)->groups);
	(*list)->groups = NULL;
    }
    (*list)->group_count = 0;


    if ((*list)->addrs) {
	int i;

	for (i = 0; i < (*list)->addr_count; i++) 
	    if ((*list)->addrs[i].addr)
		free_address(& ((*list)->addrs[i].addr));
	
	free((*list)->addrs);
	(*list)->addrs = NULL;
    }
    (*list)->addr_count = 0;
    (*list)->alloced    = 0;

    (*list)->magic = 0;      /* invalidate */
    free(*list);
    *list = NULL;
}

static void zero_addr_list_item P_((struct addr_list_item *p));
static void zero_addr_list_item(p)
     struct addr_list_item *p;
{
    p->groupidx = -1;
    p->addr     = NULL;
}

struct addr_list * new_empty_addr_list(alloced)
     int alloced;
{
    struct addr_list *ret =  safe_zero_alloc(sizeof(*ret));

    ret->magic = ADDR_list_magic;
    
    ret->group_count = 0;
    ret->groups      = NULL;

    ret->addrs      = NULL;
    ret->addr_count = 0;
    ret->alloced    = 0;
    if (alloced > 0) {
	int i;
	    
	ret->addrs = safe_calloc(alloced, 
				 sizeof(ret->addrs[0]));

	for (i = 0; i < alloced; i++) {
	    zero_addr_list_item(& (ret->addrs[i]));
	}
	ret->alloced = alloced;
    }

    return ret;
}

void append_addr_list(list,src)
     struct addr_list *list;
     const struct addr_list *src;
{
    int group_offset = 0;

    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"append_addr_list",
	      "Bad magic number (target list)",0);

    if (ADDR_list_magic != src->magic)
	panic("URL PANIC",__FILE__,__LINE__,"append_addr_list",
	      "Bad magic number (source list)",0);

    if (src->addr_count < 1 &&
	src->group_count < 1)
	return;

    group_offset = list->group_count;

    if (src->groups && 
	src->group_count > 0) {
	int i;


	list->groups = safe_array_realloc(list->groups,
					  ( group_offset + src->group_count ),
					  sizeof (list->groups[0]));
				        
	for (i = 0; i < src->group_count; i++) {

	   list->groups[list->group_count] =
	       dup_string(src->groups[i]);
	   
	   list->group_count++;
	}

    }


    if (src->addrs && 
	src->addr_count > 0) {
	int i;

	if (list->alloced < list->addr_count + src->addr_count) {
	    int size =  list->addr_count + src->addr_count;

	    list->addrs = safe_array_realloc(list->addrs,
					     size,
					     sizeof (list->addrs[0]));

	    list->alloced = size;
	}

	for (i = 0; i < src->addr_count; i++) {

	    if (list->addr_count >= list->alloced)
		panic("URL PANIC",__FILE__,__LINE__,"append_addr_list",
		      "Overflow",0);

	    if (src->addrs[i].groupidx >= 0) {
		int newidx = src->addrs[i].groupidx + group_offset;

		if (newidx < 0 || newidx >= list->group_count)
		    panic("URL PANIC",__FILE__,__LINE__,"append_addr_list",
			  "Bad group index",0); 
		
		list->addrs[list->addr_count].
		    groupidx = newidx;

	    } else
		list->addrs[list->addr_count].
		    groupidx = -1;

	    list->addrs[list->addr_count].addr =
		dup_address(src->addrs[i].addr);
	    
	    list->addr_count++;
	}
    }
}

struct addr_list * dup_addr_list(list)
     const struct addr_list *list;
{
    struct addr_list *result;

    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"dup_addr_list",
	      "Bad magic number (target list)",0);

    result = new_empty_addr_list(list->addr_count);

    append_addr_list(result,list);

    return result;
}

int add_address_to_list(list,addr,groupidx)
     struct addr_list *list;
     const struct address *addr;
     int groupidx;        /* -1 == no group */
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"add_address_to_list",
	      "Bad magic number",0);

    if (groupidx < -1 || groupidx >= list->group_count)
	panic("URL PANIC",__FILE__,__LINE__,"add_address_to_list",
	      "Bad group index",0);

    if (list->alloced < list->addr_count + 1) {
	int size = list->addr_count + 5;

	list->addrs = safe_array_realloc(list->addrs,
					 size,
					 sizeof (list->addrs[0]));
	list->alloced = size;
    }

    list->addrs[list->addr_count].groupidx = groupidx;
    list->addrs[list->addr_count].addr     = dup_address(addr);
    
    return list->addr_count++;
}

/* Returns groupidx which can be used with add_address_to_list */
int add_group_to_list(list,phrase)
     struct addr_list *list;
     const struct string *phrase;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"add_group_to_list",
	      "Bad magic number",0);

    list->groups = safe_array_realloc(list->groups,
				      ( list->group_count +1 ),
				      sizeof (list->groups[0]));

    list->groups[list->group_count] = dup_string(phrase);
   
    return list->group_count++;
}

/* Return -1 if group not found */
int addr_list_lookup_group(list,group,create)
     struct addr_list *list;
     const struct string *group;
     int create;
{
    int idx;

    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_lookup_group",
	      "Bad magic number",0);

    for (idx = 0; idx < list->group_count; idx++) {
	if (0 == string_cmp(list->groups[idx],group,-999)) 
	    return idx;
    }

    if (create)
	return add_group_to_list(list,group);	

    return -1;
}


int addr_list_item_count(list)
     const struct addr_list *list;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_item_count",
	      "Bad magic number",0);

    return list->addr_count;
}

int addr_list_group_count(list)
     const struct addr_list *list;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_group_count",
	      "Bad magic number",0);


    return list->group_count;
}

/* *group_num == -1 if not part of group */
const struct address * addr_list_get_item(list,idx,group_num)
     const struct addr_list *list;
     int idx;
     int *group_num;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_get_item",
	      "Bad magic number",0);

    if (idx < 0 || idx >= list->addr_count)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_get_item",
	      "Bad index",0);
	
    if (group_num)
	*group_num = list->addrs[idx].groupidx;

    return list->addrs[idx].addr;
}

void addr_list_replace_item(list,idx,addr,group_num)
     struct addr_list *list;
     int idx;
     const struct address *addr;
     int group_num;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_replace_item",
	      "Bad magic number",0);

    if (idx < 0 || idx >= list->addr_count)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_replace_item",
	      "Bad index",0);
    
    if (group_num < -1 || group_num >= list->group_count)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_replace_item",
	      "Bad group index",0);
    
    list->addrs[idx].groupidx = group_num;

    if (addr != list->addrs[idx].addr) {
        free_address(& (list->addrs[idx].addr));
       
	list->addrs[idx].addr     = dup_address(addr);    
    }
}


/* Give group phrase */
const struct string * addr_list_get_group(list,idx)
     const struct addr_list *list;
     int idx;
{
    if (ADDR_list_magic != list->magic)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_get_group",
	      "Bad magic number",0);

    if (idx < 0 || idx >= list->group_count)
	panic("URL PANIC",__FILE__,__LINE__,"addr_list_get_group",
	      "Bad index",0);

    return list->groups[idx];
}




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
